import math
import matplotlib.pyplot as plt

# Conversion functions
def convert_gal_mile_to_l_km(gal_per_mile):
    liters_per_km = gal_per_mile * (3.78541 / 1.60934)
    return liters_per_km

def convert_lb_to_kg(lb):
    kg = lb * 0.453592
    return kg

# Landing gear functions
def calculate_fuel_consumption_landing_gear(mlw_lb, distance_per_flight, num_flights_per_year, lifetime):
    landing_gear_percentage = 0.04
    landing_gear_weight = landing_gear_percentage * mlw_lb
    F_gal_per_mile = 1.62 + 0.0098 * (mlw_lb / 1000)
    F_liters_per_km = convert_gal_mile_to_l_km(F_gal_per_mile)
    Fuseaircraft = distance_per_flight * num_flights_per_year * lifetime * F_liters_per_km
    landing_gear_fuel_consumption = Fuseaircraft * (landing_gear_weight / mlw_lb)
    return landing_gear_fuel_consumption

def calculate_eol_costs_landing_gear(mlw_lb):
    mlw_kg = convert_lb_to_kg(mlw_lb)
    landing_gear_percentage = 0.04
    landing_gear_weight = landing_gear_percentage * mlw_kg
    wrecycling = 0.50 * landing_gear_weight
    wlandfill = 0.30 * landing_gear_weight
    wincineration = 0.20 * landing_gear_weight
    cost_recycling = -0.75
    cost_landfill = 0.2
    cost_incineration = 1
    total_eol_cost = (wrecycling * cost_recycling +
                      wlandfill * cost_landfill +
                      wincineration * cost_incineration)
    return total_eol_cost, wrecycling, wlandfill, wincineration

# Manufacturing cost calculation
def calculate_manufacturing_cost(mlw_lb, distance_per_flight_km, num_flights_per_year):
    # Constants for cost factors (hypothetical values)
    base_material_cost_per_lb = 5.0  # Euros per pound
    base_labor_cost_per_hour = 50.0  # Euros per hour
    machine_cost_per_hour = 100.0  # Euros per hour
    development_cost = 20000.0  # Euros
    overhead_factor = 1.2  # 20% overhead
    
    # Assumed parameters for production
    weight_factor = 0.04  # Assumed percentage of MLW for landing gear weight
    labor_hours_per_unit_weight = 2.0  # Hours required per unit weight (lb)
    machine_hours_per_unit_weight = 1.0  # Machine hours per unit weight (lb)

    # Calculate the estimated weight of the landing gear
    landing_gear_weight_lb = mlw_lb * weight_factor

    # Material cost calculation
    material_cost = landing_gear_weight_lb * base_material_cost_per_lb

    # Labor cost calculation
    labor_cost = landing_gear_weight_lb * labor_hours_per_unit_weight * base_labor_cost_per_hour

    # Machine cost calculation
    machine_cost = landing_gear_weight_lb * machine_hours_per_unit_weight * machine_cost_per_hour

    # Total production cost before overhead
    total_production_cost = material_cost + labor_cost + machine_cost + development_cost

    # Adjusting cost based on flight distance and frequency (hypothetical model)
    distance_factor = 1 + (distance_per_flight_km / 10000)
    frequency_factor = 1 + (num_flights_per_year / 1000)

    # Total manufacturing cost including overhead and adjustment factors
    total_manufacturing_cost = total_production_cost * distance_factor * frequency_factor * overhead_factor

    return {
        "Material Cost": material_cost,
        "Labor Cost": labor_cost,
        "Machine Cost": machine_cost,
        "Development Cost": development_cost,
        "Total Manufacturing Cost": total_manufacturing_cost
    }

# Total emissions function
def calculate_total_emissions(manufacturing_percentage, operations_percentage, fuel_production_percentage, infrastructure_operation_percentage, infrastructure_construction_percentage, gCO2tot, num_passengers, distance_per_flight):
    manufacturing = manufacturing_percentage * gCO2tot * num_passengers * distance_per_flight
    operations = operations_percentage * gCO2tot * num_passengers * distance_per_flight
    fuel_production = fuel_production_percentage * gCO2tot * num_passengers * distance_per_flight
    infrastructure_operation = infrastructure_operation_percentage * gCO2tot * num_passengers * distance_per_flight
    infrastructure_construction = infrastructure_construction_percentage * gCO2tot * num_passengers * distance_per_flight
    total_emissions = manufacturing + operations + fuel_production + infrastructure_operation + infrastructure_construction
    return manufacturing, operations, fuel_production, infrastructure_operation, infrastructure_construction, total_emissions

# Funzione per calcolare e moltiplicare i valori in base al parametro D
def calculate_values_and_multiply(num_kilometers, total_emissions):
    parameter_D = num_kilometers / 1000

    CO2 = 1.0
    NOx = 2.3 * math.atan(3.1 * parameter_D) - 2.0
    CiC = 1.1 * math.atan(0.5 * parameter_D)
    H2O = 0.2 * math.atan(parameter_D)

    CO2_multiplied = CO2 * total_emissions
    NOx_multiplied = NOx * total_emissions
    CiC_multiplied = CiC * total_emissions
    H2O_multiplied = H2O * total_emissions

    # Somma di tutti i valori moltiplicati
    total_sum = CO2_multiplied + NOx_multiplied + CiC_multiplied + H2O_multiplied

    return CO2_multiplied, NOx_multiplied, CiC_multiplied, H2O_multiplied, total_sum

# Landing gear materials analysis function (parametric version)
def analyze_landing_gear_materials(landing_gear_weight, component_proportions, material_compositions):
    total_materials = {}
    
    # Loop through each component (Main Struts, Brake Systems, Wheels)
    for component, materials in material_compositions.items():
        # Calculate Component Weight: Total Landing Gear Weight * Component Proportion
        component_weight = landing_gear_weight * component_proportions[component]
        
        # Calculate Material Weight for each material in the component
        for material, percentage in materials.items():
            weight_material = component_weight * percentage  # Weight_material = Component Weight * Material Percentage

            # Add to total materials
            if material not in total_materials:
                total_materials[material] = 0
            total_materials[material] += weight_material

    return total_materials

# Function to calculate emissions for materials (parametric version)
def calculate_emissions_for_materials(materials_weights, emissions_factors):
    total_emissions_EoL = 0
    materials_emissions = {}

    # Loop through each material and calculate emissions
    for material, weight in materials_weights.items():
        emissions_material = weight * emissions_factors[material]  # Emissions_material = Weight_material * Emissions Factor_material
        materials_emissions[material] = emissions_material
        total_emissions_EoL += emissions_material

    return total_emissions_EoL, materials_emissions

# Main function
def main():
    gCO2tot = float(input("Enter the number of grams of CO2 emitted (gCO2tot): "))
    num_passengers = int(input("Enter the number of passengers: "))
    mlw_lb = float(input("Enter the Maximum Landing Weight (MLW) in lb: "))
    distance_per_flight = float(input("Enter the distance per flight in kilometers: "))
    num_flights_per_year = int(input("Enter the number of flights per year: "))
    lifetime = 25
    fuel_price_per_liter = 0.224  # euros per liter of Kerosene

    # Convert MLW to kg
    mlw_kg = convert_lb_to_kg(mlw_lb)

    # Calculate the weight of the landing gear (4% of MLW)
    landing_gear_weight = 0.04 * mlw_kg

    # Component proportions and material compositions (parametric)
    component_proportions = {'Main Struts': 0.5, 'Brake Systems': 0.3, 'Wheels': 0.2}
    material_compositions = {
        'Main Struts': {'Titanium': 0.7, 'Aluminium': 0.2, 'High strength steel': 0.1},
        'Brake Systems': {'Carbon composites': 0.6, 'Stainless steel': 0.3, 'Aluminium': 0.1},
        'Wheels': {'Aluminium': 0.8, 'High strength steel': 0.1, 'Carbon composites': 0.1}
    }
    emissions_factors = {
        'Aluminium': 2,  # kgCO2/kg
        'Carbon composites': 3,  # kgCO2/kg
        'High strength steel': 1.5,  # kgCO2/kg
        'Titanium': 4,  # kgCO2/kg
        'Stainless steel': 1.5  # kgCO2/kg
    }

    # Analyze landing gear materials and calculate weights
    materials_weights = analyze_landing_gear_materials(landing_gear_weight, component_proportions, material_compositions)

    # Calculate emissions for landing gear materials and end of life
    total_emissions_end_of_life, materials_emissions = calculate_emissions_for_materials(materials_weights, emissions_factors)

    # Print material weights
    print("\nMaterial Weights for Landing Gear (kg):")
    for material, weight in materials_weights.items():
        print(f"{material}: {weight:.2f} kg")

    # Print material emissions
    print("\nMaterial Emissions for Landing Gear (kg CO2):")
    for material, emissions in materials_emissions.items():
        print(f"{material}: {emissions:.2f} kg CO2")

    # Print total EoL emissions
    print(f"\nTotal Emissions for the End of Life (EoL) Phase: {total_emissions_end_of_life:.2f} kg CO2")

    # Calculate landing gear total cost (use phase cost)
    landing_gear_total_cost = calculate_fuel_consumption_landing_gear(mlw_lb, distance_per_flight, num_flights_per_year, lifetime) * fuel_price_per_liter
    print(f"\nLanding Gear Use Phase Cost: {landing_gear_total_cost:.2f} euros")

    # Calculate end of life costs for landing gear
    total_eol_cost, wrecycling, wlandfill, wincineration = calculate_eol_costs_landing_gear(mlw_lb)
    print(f"Landing Gear End Of Life Cost: {total_eol_cost:.2f} euros")

    # Calculate manufacturing cost
    manufacturing_costs = calculate_manufacturing_cost(mlw_lb, distance_per_flight, num_flights_per_year)
    total_manufacturing_cost = manufacturing_costs["Total Manufacturing Cost"]
    print("\nLanding Gear Manufacturing Cost:")
    for cost_type, cost in manufacturing_costs.items():
        print(f"{cost_type}: €{cost:.2f}")

    # Emission breakdown
    manufacturing_percentage = 3.52 / 100
    operations_percentage = 78.63 / 100
    fuel_production_percentage = 2.07 / 100
    infrastructure_operation_percentage = 3.5 / 100
    infrastructure_construction_percentage = 12.28 / 100

    manufacturing, operations, fuel_production, infrastructure_operation, infrastructure_construction, total_emissions = calculate_total_emissions(
        manufacturing_percentage, operations_percentage, fuel_production_percentage, infrastructure_operation_percentage, infrastructure_construction_percentage,
        gCO2tot, num_passengers, distance_per_flight)

    print("\nEmissions Breakdown:")
    print(f"Manufacturing: {manufacturing:.2f} g CO2")
    print(f"Operations: {operations:.2f} g CO2")
    print(f"Fuel Production: {fuel_production:.2f} g CO2")
    print(f"Infrastructure Operation: {infrastructure_operation:.2f} g CO2")
    print(f"Infrastructure Construction: {infrastructure_construction:.2f} g CO2")
    print(f"Total Emissions: {total_emissions:.2f} g CO2")

    # Calculate and display multiplied values
    CO2_multiplied, NOx_multiplied, CiC_multiplied, H2O_multiplied, total_sum = calculate_values_and_multiply(distance_per_flight, total_emissions)

    print("\nMultiplied Values:")
    print(f"CO2 Multiplied: {CO2_multiplied:.2f} g")
    print(f"NOx Multiplied: {NOx_multiplied:.2f} g")
    print(f"CiC Multiplied: {CiC_multiplied:.2f} g")
    print(f"H2O Multiplied: {H2O_multiplied:.2f} g")
    print(f"Total Sum of Multiplied Values: {total_sum:.2f} g")

    # Landing gear fuel consumption and use phase cost
    landing_gear_fuel_consumption = calculate_fuel_consumption_landing_gear(mlw_lb, distance_per_flight, num_flights_per_year, lifetime)
    print(f"\nLanding Gear Fuel Consumption: {landing_gear_fuel_consumption:.2f} liters")

    landing_gear_total_cost = landing_gear_fuel_consumption * fuel_price_per_liter
    print(f"Landing Gear Use Phase Cost: {landing_gear_total_cost:.2f} euros")

    # Display masses of materials in EoL phase
    print("\nMasses of Materials in End of Life (EoL) Phase:")
    print(f"Recycling: {wrecycling:.2f} kg")
    print(f"Landfill: {wlandfill:.2f} kg")
    print(f"Incineration: {wincineration:.2f} kg")

    # Calculate Manufacturing and Use Operation Emissions
    mlw_kg = convert_lb_to_kg(mlw_lb)
    landing_gear_weight_kg = 0.04 * mlw_kg
    manufacturing_and_use_operation_emissions = (landing_gear_weight_kg / mlw_kg) * total_sum
    print(f"Manufacturing and Use Operation Emissions: {manufacturing_and_use_operation_emissions:.2f} g CO2")

    # Calculate total landing gear emissions contribution
    landing_gear_emissions_contribution = manufacturing_and_use_operation_emissions + (total_emissions_end_of_life)*1000
    print(f"Landing Gear Total Emissions: {landing_gear_emissions_contribution:.2f} g CO2")

    # Calculate total aircraft cost
    total_aircraft_cost = total_manufacturing_cost + landing_gear_total_cost + total_eol_cost
    print(f"\nTotal Landing Gear Cost: {total_aircraft_cost:.2f} euros")

    # Scatter plot for costs
    labels = ['Total Manufacturing Cost', 'Use Phase Cost', 'EOL Cost']
    sizes = [total_manufacturing_cost, landing_gear_total_cost, total_eol_cost]
    fig1, ax1 = plt.subplots()
    ax1.scatter(labels, sizes)
    plt.xlabel('Cost Types')
    plt.ylabel('Costs (Euros)')
    plt.title('Cost Distribution')
    plt.show()

    # Pie chart for EOL emissions
    materials = list(materials_emissions.keys())
    emissions = list(materials_emissions.values())
    fig2, ax2 = plt.subplots()
    ax2.pie(emissions, labels=materials, autopct='%1.1f%%', startangle=90)
    ax2.axis('equal')  # Equal aspect ratio ensures that pie is drawn as a circle.
    plt.title('EOL Emissions by Material')
    plt.show()

    # Bar chart for manufacturing costs
    cost_types = list(manufacturing_costs.keys())[:-1]  # Exclude Total Manufacturing Cost from individual costs
    costs = [manufacturing_costs[cost] for cost in cost_types]
    fig3, ax3 = plt.subplots()
    ax3.bar(cost_types, costs)
    plt.xlabel('Cost Type')
    plt.ylabel('Cost (Euros)')
    plt.title('Manufacturing Costs Breakdown')
    plt.xticks(rotation=45, ha='right')
    plt.tight_layout()
    plt.show()

# Run the main function
if __name__ == "__main__":
    main()

